library(LHD)
X = rLHD(n = 5, k = 3); X  #This generates a 5 by 3 random LHD, denoted as X

phi_p(X)             #The maximin L1-distance criterion.
phi_p(X, p = 10, q = 2)    #The maximin L2-distance criterion.
MaxProCriterion(X)   #The maximum projection criterion.
AvgAbsCor(X)         #The average absolute correlation criterion.
MaxAbsCor(X)         #The maximum absolute correlation criterion.


#Generate a 5 by 3 maximin distance LHD by the SA function.
try.SA = SA(n = 5, k = 3); try.SA 

phi_p(try.SA)   #\phi_p is smaller than that of a random LHD (0.3336608).


#Similarly, generations of 5 by 3 maximin distance LHD by the SA2008, LaPSO and GA functions.
try.SA2008 = SA2008(n = 5, k = 3)
try.LaPSO = LaPSO(n = 5, k = 3)
try.GA = GA(n = 5, k = 3)

#Generate an OA(9,2,3,2), an orthogonal array with 9 runs, 2 factors, 3 levels, and 2 strength.
OA = matrix(c(rep(1:3, each = 3), rep(1:3, times = 3)),
          ncol = 2, nrow = 9, byrow = FALSE)
#Generates a maximin distance LHD with the same design size as the input OA
#by the orthogonal-array-based simulated annealing algorithm.
try.OASA = OASA(OA)
OA; try.OASA

#Below try.SA is a 5 by 3 maximin distance LHD generated by the SA with 30 iterations (N = 30).
#The temperature starts at 10 (T0 = 10) and decreases 10% (rate = 0.1) each time.
#The minimium temperature allowed is 1 (Tmin = 1) and the maximum perturbations that 
#the algorithm will try without improvements is 5 (Imax = 5). The optimality criterion 
#used is maximin distance criterion (OC = "phi_p") with p = 15 and q = 1, and the 
#maximum CPU time is 5 minutes (maxtime = 5).
try.SA = SA(n = 5, k = 3, N = 30, T0 = 10, rate = 0.1, Tmin = 1, Imax = 5, OC = "phi_p",
            p = 15, q = 1, maxtime = 5); try.SA

phi_p(try.SA)    


#Below try.SA2008 is a 5 by 3 maximin distance LHD generated by SA with 
#the multi-objective optimization approach. The input arguments are interpreted 
#the same as the design try.SA above.
try.SA2008=SA2008(n=5,k=3,N=30,T0=10,rate=0.1,Tmin=1,Imax=5,
                  OC="phi_p",p=15,q=1,maxtime=5)

#Below try.OASA is a 9 by 2 maximin distance LHD generated by the
#orthogonal-array-based simulated annealing algorithm with the input 
#OA (defined previously), and the rest input arguments are interpreted the 
#same as the design try.SA above.
try.OASA = OASA(OA, N = 30, T0 = 10, rate = 0.1, Tmin = 1, Imax = 5,
                OC = "phi_p", p = 15, q = 1, maxtime = 5)

#Below try.LaPSO is a 5 by 3 maximum projection LHD generated by the particle swarm 
#optimization algorithm with 20 particles (m = 20) and 30 iterations (N = 30).
#Zero (or two) elements in any column of the current particle should be the same as
#the elements of corresponding column from personal best (or global best), because
#of SameNumP = 0 (or SameNumG = 2).
#The probability of exchanging two randomly selected elements is 0.5 (p0 = 0.5).
#The optimality criterion is maximum projection criterion (OC = "MaxProCriterion").
#The maximum CPU time is 5 minutes (maxtime = 5).
try.LaPSO = LaPSO(n = 5, k = 3, m = 20, N = 30, SameNumP = 0, SameNumG = 2,
                  p0 = 0.5, OC = "MaxProCriterion", maxtime = 5); try.LaPSO

#Recall the value is 0.5375482 from the random LHD in Section 2.
MaxProCriterion(try.LaPSO)


#Below try.GA is a 5 by 3 OLHD generated by the genetic algorithm with the 
#population size 20 (m = 20), number of iterations 30 (N = 30),  mutation 
#probability 0.5 (pmut = 0.5), maximum absolute correlation criterion
#(OC = "MaxAbsCor"), and maximum CPU time 5 minutes (maxtime = 5).
try.GA = GA(n = 5, k = 3, m = 20, N = 30, pmut = 0.5, OC = "MaxAbsCor", 
            maxtime = 5); try.GA

#Recall the value is 0.9 from the random LHD in Section 2.
MaxAbsCor(try.GA)

#Make sure both packages are properly installed before load them
library(LHD) 
library(MaxPro)

count = 0 #Define a variable for counting purpose

k = 5       #Factor size 5
n = 10*k    #Run size = 10*factor size

#Setting 500 iterations for both algorithms, below loop counts
#how many times the GA from LHD package outperforms the algorithm 
#from MaxPro package out of 100 times

for (i in 1:100) {
  
  LHD = LHD::GA(n = n, k = k, m = 100, N = 500)
  MaxPro = MaxPro::MaxProLHD(n = n, p = k, total_iter = 500)$Design
  
  #MaxPro * n + 0.5 applied the transformation mentioned in Section 2 
  #to revert the scaling.
  Result.LHD = LHD::MaxProCriterion(LHD)
  Result.MaxPro = LHD::MaxProCriterion(MaxPro * n + 0.5)
  
  if (Result.LHD < Result.MaxPro) {count = count + 1}
  
}

count

#FastMmLHD(8, 8) generates an optimal 8 by 8 maximin L_1 distance LHD.
try.FastMm = FastMmLHD(n = 8, k = 8); try.FastMm

#OA2LHD(OA) expands an input OA to an LHD of the same run size.
try.OA2LHD = OA2LHD(OA)
OA; try.OA2LHD

#OLHD.Y1998(m = 3) generates a 9 by 4 orthogonal LHD.
#Note that 2^m+1 = 9 and 2*m-2 = 4.
try.Y1998 = OLHD.Y1998(m = 3); try.Y1998

MaxAbsCor(try.Y1998)    #column-wise correlations are 0.

#OLHD.C2007(m = 4) generates a 17 by 7 orthogonal LHD.
#Note that 2^m+1 = 17 and $4+{4-1 \choose 2}$ = 7.
try.C2007 = OLHD.C2007(m = 4); dim(try.C2007)

MaxAbsCor(try.C2007)    #column-wise correlations are 0


#OLHD.S2010(C = 3, r = 3, type = "odd") generates a 49 by 8 orthogonal LHD.
#Note that 3*2^4+1 = 49 and 2^3 = 8.
dim(OLHD.S2010(C = 3, r = 3, type = "odd"))

MaxAbsCor(OLHD.S2010(C = 3, r = 3, type = "odd")) #column-wise correlations are 0


#OLHD.S2010(C = 3, r = 3, type = "even") generates a 48 by 8 orthogonal LHD.
#Note that 3*2^4 = 48 and 2^3 = 8.
dim(OLHD.S2010(C = 3, r = 3, type = "even"))

MaxAbsCor(OLHD.S2010(C = 3, r = 3, type = "even")) #column-wise correlations are 0


#Create a 5 by 2 OLHD.
OLHD = OLHD.C2007(m = 2)

#Create an OA(25, 6, 5, 2).
OA = matrix(c(2,2,2,2,2,1,2,1,5,4,3,5,3,2,1,5,4,5,1,5,4,3,2,5,4,1,3,5,2,3,
                1,2,3,4,5,2,1,3,5,2,4,3,1,1,1,1,1,1,4,3,2,1,5,5,5,5,5,5,5,1,4,4,4,4,4,1,
                3,1,4,2,5,4,3,3,3,3,3,1,3,5,2,4,1,3,3,4,5,1,2,2,5,4,3,2,1,5,2,3,4,5,1,2,
                2,5,3,1,4,4,1,4,2,5,3,4,4,2,5,3,1,4,2,4,1,3,5,3,5,3,1,4,2,4,5,2,4,1,3,3,
                5,1,2,3,4,2,4,5,1,2,3,2), ncol = 6, nrow = 25, byrow = TRUE)

#OLHD.L2009(OLHD, OA) generates a 25 by 12 orthogonal LHD.
#Note that n = 5 so n^2 = 25. p = 2 and f = 3 so 2fp = 12.
dim(OLHD.L2009(OLHD, OA))

MaxAbsCor(OLHD.L2009(OLHD, OA))    #column-wise correlations are 0.


#OLHD.B2001(n = 11, k = 5) generates a 11 by 5 orthogonal LHD.
dim(OLHD.B2001(n = 11, k = 5))





